/*
  ULNStepper.cpp - Library to use with controller ULN2003
				   for stepper motors.
				
  Created by A. Carpini, 24 April 2015.
  Contact: alessio.carpini@gmail.com
  
*/

#include "Arduino.h"
#include "ULNStepper.h"

/* Costructor Class 
	- define the number of steps for 360° rotation (steps)
	- define digital Output pin (pin1, pin2, pin3, pin4)
*/
ULNStepper::ULNStepper(uint16_t steps, uint8_t pin1, uint8_t pin2, uint8_t pin3, uint8_t pin4)
{
	pinMode(pin1, OUTPUT);
	pinMode(pin2, OUTPUT);
	pinMode(pin3, OUTPUT);
	pinMode(pin4, OUTPUT);
	_pin1 = pin1;
	_pin2 = pin2;
	_pin3 = pin3;
	_pin4 = pin4;
	
	_revsteps = steps;
	_currentstep = 0;
	
}

/* Set the rpm of the stepper motor

*/	
void ULNStepper::setSpeed(uint16_t rpm)
{
	// Periodo in us di ogni passo per garantire quel rpm
	us_per_step = 60000000 / ((uint32_t)_revsteps * (uint32_t)rpm);
	steppingcounter = 0;
   
}

/* Use this function to move the motor of "steps" number of steps
	- steps: number of steps
	- dir: direction of the moviment (FORWARD or REVERSE)
	- style: type of the moviment (SINGLE, DOUBLE, HALF)

*/
void ULNStepper::step(uint16_t steps, uint8_t dir,  uint8_t style)
{
	uint32_t uspers = steps;
	
	if (style == HALF) 
	{
		// Perché devo fare il doppio degli step per fare una rotazione
		uspers /= 2;
	}
	
	while(steps--)
	{
		onestep(dir, style);
		delay(uspers/1000);
		
		// Correzione del ritardo in caso uspers/1000 non sia intero
		steppingcounter += (uspers % 1000);
		if (steppingcounter >= 1000) 
		{
			delay(1);
			steppingcounter -= 1000;
		}	
	}
}

/* Function to move the motor of 1 step
	- dir: direction of the moviment
	- style: type of the moviment
*/
uint8_t ULNStepper::onestep(uint8_t dir, uint8_t style)
{	
	// -- Aggiorno lo stato --
	// SINGLE: only one coil is active at the same time
	if (style == SINGLE) 
	{	
		// Se sono a step pari (0, 2, 4, 6)
		if ( !(_currentstep % 2))
		{ 
			if (dir == FORWARD)
			{
				_currentstep += 2;
				
			}
			else
			{
				_currentstep -= 2;
			}			
		}
		else
		{
			// Se sono in step dispari (1, 3, 5, 7)
			if (dir == FORWARD)  
			{
				_currentstep += 1;
			}
			else
			{
				_currentstep -= 1;
			}
		}
	}
	// DOUBLE: two coils are active at the same time
	else if (style == DOUBLE)
	{
		// Se sono in step dispari (1, 3, 5, 7)
		if ( (_currentstep % 2) )
		{ 
			if (dir == FORWARD) 
			{
				_currentstep += 2;
			}	 
			else 
			{	
				_currentstep -= 2;
			}
		} 
		else
		{
			// Se sono in step pari (0, 2, 4, 6)
			if (dir == FORWARD)
			{
				_currentstep += 1;
			} 
			else
			{
				_currentstep -= 1;
			}
		}
	} 
	// HALF: two coils are active at the same time only during rotation
	else if (style == HALF) 
	{
		if (dir == FORWARD) 
		{
			_currentstep += 1;
		} 
		else
		{
			_currentstep -= 1;
		}	
	}
	else  
	{
		return 0;
	}
	
	_currentstep += 8;
	_currentstep %= 8;
	
	
	
	switch(_currentstep)
	{
		case 0:
			digitalWrite(_pin1, HIGH);
			digitalWrite(_pin2, LOW);
			digitalWrite(_pin3, LOW);
			digitalWrite(_pin4, LOW);
			break;
		case 1:
			digitalWrite(_pin1, HIGH);
			digitalWrite(_pin2, HIGH);
			digitalWrite(_pin3, LOW);
			digitalWrite(_pin4, LOW);
			break;
		case 2:
			digitalWrite(_pin1, LOW);
			digitalWrite(_pin2, HIGH);
			digitalWrite(_pin3, LOW);
			digitalWrite(_pin4, LOW);
			break;
		case 3:
			digitalWrite(_pin1, LOW);
			digitalWrite(_pin2, HIGH);
			digitalWrite(_pin3, HIGH);
			digitalWrite(_pin4, LOW);
			break;
		case 4:
			digitalWrite(_pin1, LOW);
			digitalWrite(_pin2, LOW);
			digitalWrite(_pin3, HIGH);
			digitalWrite(_pin4, LOW);
			break;
		case 5:
			digitalWrite(_pin1, LOW);
			digitalWrite(_pin2, LOW);
			digitalWrite(_pin3, HIGH);
			digitalWrite(_pin4, HIGH);
			break;
		case 6:
			digitalWrite(_pin1, LOW);
			digitalWrite(_pin2, LOW);
			digitalWrite(_pin3, LOW);
			digitalWrite(_pin4, HIGH);
			break;
		case 7:
			digitalWrite(_pin1, HIGH);
			digitalWrite(_pin2, LOW);
			digitalWrite(_pin3, LOW);
			digitalWrite(_pin4, HIGH);
			break;
	}
	
	return _currentstep;
}
	
	